
import { camelCase, cleanComponentProperties, generate } from "./cleanFunc";
import { generateMappedValue } from "./components/commonFunc";
import { paddingDivider } from "./components/frameFunc";
import { panelResult } from "./components/panel";
import { generateTextCustomAttribute, textTagTypeFunc } from "./components/textLink";
import { businessComponentsFunc, coreComponentsFunc, frameWorkSpectraComponentsFunc } from "./fetchComponentList";
import { childFlexType, parentFlexType } from "./flexFunc";
import { componentToComponentMapping, idToComponentMapping, idToComponentMapping25A, idToKeyMapping24D, idToKeyMapping25A } from "./maps/componentMap";
import { componentMap } from "./maps/exportMap";
import { checkIgnoreKey, removeIgnoreKey } from "./pluginData";
import { businessComponents, buttonModalReaction, coreComponents, frameWorkComponents, getComponentJson, spectraComponent } from "./toolkitComponents";

type CustomAttributeRule = {
    value: string | boolean | ((node: any) => string | boolean);
    attrKey: string;
    attrValue: string | ((node: any) => string);
};

// Type for the customAttribute structure within each component
type CustomAttributeMap = {
    [property: string]: CustomAttributeRule[];
};

export let businessComponentsList: any;
let frameWorkComponentsList: any;
let spectraComponentsList: any;
let coreComponentsList: any;

const addUserAssistance = (node: any, output: any) => {
    if (!node || !output) return;

    const cleanedProps = cleanComponentProperties(node.componentProperties);
    if (cleanedProps['userassistance']) {
        const userAssistanceNode = node.children.find((child: any) => child.name === "Private/❗️User Assistance");
        if (userAssistanceNode) {
            const userAssistanceCleanedProps = cleanComponentProperties(userAssistanceNode.componentProperties);

            if (userAssistanceCleanedProps.type === 'Required') {
                output.isRequiredField = true;
            }
        }
    }
    return;
};

export function traverseNode(node: any, isIgnoredRoot: boolean = true, isFlexChild: boolean = false) {

    if (!node) return {};
    if (node.visible === false || checkIgnoreKey(node.id)) {
        removeIgnoreKey(node.id);
        return {};
    }

    const componentName = camelCase(node.name);
    const flexData = node.getPluginData("flexWidth");


    if (node.type === 'FRAME') {
        const frameResult = frameNode(node, isFlexChild);

        if (Object.keys(frameResult).length > 0) {
            if (componentName === 'ojPanel') {
                panelResult(node, frameResult);
            }
        }
        return frameResult;
    }


    if (node.type === 'TEXT') {
        const textResult: any =
        {
            "figmaId": node.id,
            "componentID": `'${generate()}'`,
            "type": "Text",
            "id": generate(),
            "nameholder": node.characters,
            "mappedValue": generateMappedValue(node),
            "use": "ObservableVariable",
            "textBindingOption": "NLS",
            "customAttribute": generateTextCustomAttribute(node),
            "tagType": textTagTypeFunc(node)
        }

        if (flexData) {
            textResult.customAttribute = processFlexData(flexData, textResult.customAttribute);
        } else if (node.parent?.layoutMode !== 'NONE' && isFlexChild) {
            const flexType = childFlexType(node);
            if (flexType) {
                const flexItem = {
                    colon: true,
                    attribute: "class",
                    value: `'oj-flex-item${flexType}'`
                };
                textResult.customAttribute = addAttributes(textResult.customAttribute, flexItem);
            }
        }

        return [textResult];
    }

    let id: string | null = null;

    if (node.type === 'INSTANCE') {
        const mainComponent = node.mainComponent;
        const componentSet = mainComponent?.parent;

        id = (componentSet?.type === 'COMPONENT_SET') ? componentSet.key : mainComponent?.key || null;
    }

    if (id) {
        const compsName: string | undefined = idToKeyMapping24D(id) || idToKeyMapping25A(id);

        if (compsName) {
            const mapData = componentMap[compsName];
            if (mapData) {
                let output: Record<string, any> = {
                    figmaId: node.id,
                    componentID: `'${generate()}'`,
                    type: mapData.type || componentName,
                    ...extractProperties(node, mapData)
                };

                processConditionalExpression(node, output);

                if (mapData.customAttribute) {
                    const customAttributes = generateCustomAttributes(node, mapData.customAttribute);
                    if (customAttributes.length > 0) {
                        output.customAttribute = customAttributes;
                    }
                }

                // Add extra properties if defined in the map
                if (mapData.extraProperties) {
                    Object.assign(output, mapData.extraProperties(node));
                }

                if (mapData.connectedNodes) {
                    const connectedNodes = mapData.connectedNodes(node);
                    if (connectedNodes.length > 0) {
                        connectedNodes.unshift(output);
                        return connectedNodes;
                    }
                }

                addUserAssistance(node, output);

                if (flexData) {
                    output.customAttribute = processFlexData(flexData, output.customAttribute);
                } else if (node.parent?.layoutMode !== 'NONE' && isFlexChild && componentName !== 'columnShare') {
                    const flexType = childFlexType(node);
                    if (flexType) {
                        const flexItem = {
                            colon: true,
                            attribute: "class",
                            value: `'oj-flex-item${flexType}'`
                        };
                        output.customAttribute = addAttributes(output.customAttribute, flexItem);
                    }
                }

                if (compsName === "button") {
                    buttonModalReaction(node);
                }

                return output;
            }
        }

    }

    // const createToolkiTFrameResult = (result: any) => ({
    //     type: "Container",
    //     componentID: `'${generate()}'`,
    //     figmaId: `frame${node.id}`,
    //     containerType: 'NormalContainer',
    //     children: [result],
    //     customAttribute: processFlexData(flexData, result?.customAttribute || [])
    // });

    const createToolkiTFrameResult = (result: any) => {
        let customAttribute = result?.customAttribute || [];

        if (flexData) {
            customAttribute = processFlexData(flexData, customAttribute);
        } else if (node.parent?.layoutMode !== 'NONE' && isFlexChild) {
            const flexType = childFlexType(node);

            if (flexType) {
                const flexItem = {
                    colon: true,
                    attribute: "class",
                    value: `'oj-flex-item${flexType}'`
                };
                customAttribute = addAttributes(customAttribute, flexItem);
            }
        }

        return {
            type: "Container",
            componentID: `'${generate()}'`,
            figmaId: `frame${node.id}`,
            containerType: 'NormalContainer',
            children: [result],
            customAttribute
        };
    };

    const componentJson =
        getComponentJson(coreComponentsList, componentName)
        || getComponentJson(frameWorkComponentsList, componentToComponentMapping(componentName))
        || getComponentJson(businessComponentsList, componentName)
        || getComponentJson(spectraComponentsList, componentName);

        if (componentJson) {
        const result = (getComponentJson(coreComponentsList, componentName))
            ? coreComponents(componentJson, node)
            : (getComponentJson(frameWorkComponentsList, componentToComponentMapping(componentName)))
                ? frameWorkComponents(componentName, componentJson, node)
                : (getComponentJson(businessComponentsList, componentName))
                    ? businessComponents(componentJson, node)
                    : spectraComponent(componentJson, node)

        if (flexData || (node.parent?.layoutMode !== 'NONE' && isFlexChild)) {
            const frameResult = createToolkiTFrameResult(result);
            processConditionalExpression(node, frameResult);
            return frameResult;
        }

        processConditionalExpression(node, result);
        return result;
    }

    if (node.fills && node.fills?.length > 0 && node.fills[0]?.type === 'IMAGE') {
        const flexData = node.getPluginData("flexWidth");
        let customAttribute;
        if (flexData) {
            customAttribute = processFlexData(flexData, []) || [];
        } else if (node.parent?.layoutMode !== 'NONE' && isFlexChild) {
            const flexType = childFlexType(node);

            if (flexType) {
                const flexItem = {
                    colon: true,
                    attribute: "class",
                    value: `'oj-flex-item${flexType}'`
                };
                customAttribute = addAttributes(customAttribute, flexItem) || [];
            }
        }
        const imageResult: any = {
            nameholder: `'${node.name}'`,
            type: "Image",
            componentID: `'${generate()}'`,
            figmaId: node.id,
            customAttribute
        };
        processConditionalExpression(node, imageResult);
        return imageResult;
    }

    if (node.type === 'INSTANCE') {

        const flexParent = node.layoutMode !== 'NONE';

        function processChildren(node: any): any[] {
            return (node.children || [])
                .map((childNode: any) => traverseNode(childNode, false, flexParent))
                .flat()
                .filter((child: any) => child && child.type);
        }

        const children = processChildren(node);
        const knownComponentExists = children.length > 0;

        if (isIgnoredRoot) {
            figma.ui.postMessage({
                ignoredInstance: { name: node.name, id: node.id, knownComponentExists }
            });
        }

        if (knownComponentExists) {
            function wrapContainer(node: any, children: any[]): any {
                //changed for flex
                // const output: any = {
                //     type: "Container",
                //     componentID: `'${generate()}'`,
                //     figmaId: node.id,
                //     containerType: "NormalContainer",
                //     children: children.map(child =>
                //         child.type === "Container" && child.children.length > 0 && child.containerType === 'NormalContainer'
                //             ? wrapContainer(child, child.children)
                //             : child
                //     )
                // };

                const output: any = {
                    type: "Container",
                    componentID: `'${generate()}'`,
                    figmaId: node.id,
                    containerType: "NormalContainer",
                    children: [...children]
                };

                // const flex = node.getPluginData("flexExists");
                // if (flex === 'yes') {
                //     const flexAttr = {
                //         colon: true,
                //         attribute: "class",
                //         value: `'oj-flex oj-flex-items-pad'`
                //     };
                //     output.customAttribute = addAttributes(output.customAttribute, flexAttr);
                // } else
                if (node.layoutMode !== 'NONE') {
                    const flexType = parentFlexType(node);
                    const flexAttr = {
                        colon: true,
                        attribute: "class",
                        value: `'oj-flex${flexType}'`
                    };
                    output.customAttribute = addAttributes(output.customAttribute, flexAttr);
                }

                // const flexData = node.getPluginData("flexWidth");
                // if (flexData) {
                //     output.customAttribute = processFlexData(flexData, output.customAttribute);
                // } else 
                if (node.parent?.layoutMode !== 'NONE' && isFlexChild) {
                    const flexType = childFlexType(node);
                    if (flexType) {
                        const flexItem = {
                            colon: true,
                            attribute: "class",
                            value: `'oj-flex-item${flexType}'`
                        };
                        output.customAttribute = addAttributes(output.customAttribute, flexItem);
                    }
                }

                return output;
            }

            return wrapContainer(node, children);
        }
        return {};
    }


    console.log('Invalid component', node.name, node.type);
    return {};
}

function extractProperties(node: any, mapData: any): Record<string, any> {
    const output: Record<string, any> = {};
    const cleanedProps = cleanComponentProperties(node.componentProperties);
    for (const prop in mapData.properties) {
        const mapProp = mapData.properties[prop];
        const cleanValue = cleanedProps[prop];
        output[mapProp.keyName] = typeof mapProp.value === 'function'
            ? mapProp.value(cleanValue, node)
            : mapProp.value !== undefined
                ? mapProp.value
                : cleanValue;
    }

    return output;
}

function generateCustomAttributes(node: any, customAttributeConfig: CustomAttributeMap): Array<Record<string, string>> {
    const attributes: Array<{ attribute: string; value: string }> = [];
    const cleanedProps = cleanComponentProperties(node.componentProperties);
    for (const prop in customAttributeConfig) {
        const rules = customAttributeConfig[prop];
        const nodeValue = cleanedProps[prop];
        rules.forEach((rule: CustomAttributeRule) => {
            const ruleValue = typeof rule.value === 'function' ? rule.value(node) : rule.value;
            const attrValue = typeof rule.attrValue === 'function' ? rule.attrValue(node) : rule.attrValue;
            if (nodeValue === ruleValue || typeof rule.value === 'function') {
                if (attrValue === '') {
                    return;
                }
                attributes.push({ attribute: rule.attrKey, value: attrValue });
            }

        });
    }

    return attributes;
}

function processConditionalExpression(node: any, instanceResult: any): void {
    const conditionData = node.getPluginData("conditionData");
    if (conditionData) {
        const conditionArray = JSON.parse(conditionData);
        for (const conditionItem of conditionArray) {
            const conditionalExpression = getCondition(instanceResult, conditionItem);
            if (conditionalExpression) {
                instanceResult.conditionalExpression = conditionalExpression;
            }
        }
    }
}

export function getCondition(instanceResult: any, conditionArray: any) {
    try {
        let conditionalExpression: any;

        const processCondition = (conditionItem: any) => {
            let subCondition: string | undefined = undefined;

            if (conditionItem.controllingComponentFrom === 'component' && conditionItem.variableComponentTo === 'component') {
                if (conditionItem.controllingComponentValue) {
                    let mappedName: string;
                    const lastFiveDigits = conditionItem.controllingComponentId.replace(/:/g, '').slice(-5);

                    if (conditionItem.controllingComponentName === 'Tab Bar') {
                        mappedName = `selectedTab${lastFiveDigits}`
                    } else {
                        mappedName = `${camelCase(conditionItem.controllingComponentName).replace(/[\s-]/g, '') + lastFiveDigits}Var`
                    }
                    const value = conditionItem.controllingComponentValue?.replace(/ /g, "").toUpperCase();
                    subCondition = conditionItem.conditionType === 'hide' ? `${mappedName}()!=='${value}'` : `${mappedName}()==='${value}'`;
                }
            } else if (conditionItem.controllingComponentFrom === 'property' && conditionItem.variableComponentTo === 'component') {
                if (conditionItem.controllingProperty && conditionItem.controllingPropertyValue) {
                    const value = `$properties.${conditionItem.controllingProperty}==='${conditionItem.controllingPropertyValue}'`;
                    subCondition = conditionItem.conditionType === 'hide' ? `!${value}` : value;
                }
            }
            return subCondition;
        };

        let conditionalLogic: string;
        if (Array.isArray(conditionArray)) {
            conditionalLogic = '(' + conditionArray
                .map(processCondition)
                .filter((subCondition): subCondition is string => subCondition !== undefined)
                .join(' && ') + ')';
        } else {
            conditionalLogic = processCondition(conditionArray) || '';
        }

        if (instanceResult.conditionalExpression?.conditionalValue) {
            conditionalExpression = { conditionalValue: instanceResult.conditionalExpression.conditionalValue + ' || ' + conditionalLogic };
        } else {
            conditionalExpression = { conditionalValue: conditionalLogic };
        }
        return conditionalExpression;
    } catch (error) {
        console.error(error);
    }
}

export function frameNode(node: any, isFlexChild: boolean = false) {
    try {
        if (!node) {
            throw new Error("The 'node' parameter is missing or undefined.");
        }

        if (!node.visible) {
            return {};
        }

        const children = nodeChild(node).filter((child: any) => Object.keys(child).length > 0);

        if (children.length === 0) {
            return {};
        }

        let containerType: string = 'NormalContainer';

        if (node.name === 'oj-li') {
            containerType = 'listItem';
        }

        const output: Record<string, any> = {
            type: "Container",
            componentID: `'${generate()}'`,
            figmaId: node.id,
            containerType: containerType,
            children,
        };

        processConditionalExpression(node, output);

        const flex = node.getPluginData("flexExists");
        if (flex === 'yes') {
            const flex = {
                "colon": true,
                "attribute": "class",
                "value": `'oj-flex oj-flex-items-pad'`
            };
            output.customAttribute = addAttributes(output.customAttribute, flex);
        } else if (
            node.layoutMode !== 'NONE' &&
            node.children &&
            !node.children.some((child: { name: string }) =>
                child.name === 'Page Section' || child.name === 'Form Layout')
        ) {
            const flexType = parentFlexType(node);
            const flex = {
                "colon": true,
                "attribute": "class",
                "value": `'oj-flex${flexType}'`
            };
            output.customAttribute = addAttributes(output.customAttribute, flex);
        }

        const flexData = node.getPluginData("flexWidth");

        if (flexData) {
            output.customAttribute = processFlexData(flexData, output.customAttribute);
        } else if (node.parent?.layoutMode !== 'NONE' && isFlexChild) {

            const flexType = childFlexType(node);

            if (flexType) {
                const flexItem = {
                    colon: true,
                    attribute: "class",
                    value: `'oj-flex-item${flexType}'`
                };
                output.customAttribute = addAttributes(output.customAttribute, flexItem);
            }
        }
        paddingDivider(node, output);

        if (node.name === 'oj-drilldown') {

            const drillOutput = {
                type: "Container",
                componentID: `'${generate()}'`,
                figmaId: node.id,
                containerType: "NormalContainer",
                customAttribute: [
                    {
                        "colon": true,
                        "attribute": "class",
                        "value": "'oj-flex-sm-initial oj-flex-item oj-sm-margin-2x-start oj-listview-drill-icon'"
                    }
                ]
            }

            return drillOutput;
        }
        return output;
    } catch (error) {
        console.error("An error occurred in frameNode:", error);
        return {};
    }
}

export function nodeChild(node: any) {
    try {
        if (!node) {
            throw new Error("The 'node' parameter is missing or undefined.");
        }

        if (node.type !== 'FRAME') {
            const result = traverseNode(node);
            return result ? (Array.isArray(result) ? result : [result]) : [];
        }

        let result: any[] = [];

        for (const childNode of node.children || []) {
            if (!childNode) {
                console.warn(`Child node is null or undefined.`);
                continue;
            }

            const isFrameNode = childNode.type === 'FRAME';
            const isSpecialLayout = childNode.name === 'Form Layout' || childNode.name === 'Page Section';

            //commented for flex
            // const isVerticalFrame = node.type === 'FRAME' && node.layoutMode === 'VERTICAL' && node.name !== 'Transaction Body';

            const flexFrame = node.layoutMode !== 'NONE' && childNode.name !== 'Form Layout' && childNode.name !== 'Page Section'
            let childResult;

            // if (isFrameNode || isVerticalFrame) {
            //     childResult = frameNode(childNode);
            // } else 

            // if (isFrameNode) {
            //     childResult = frameNode(childNode);
            // }
            // else 
            if (isSpecialLayout) {
                childResult = traverseNode(childNode);
            } else if (flexFrame) {
                childResult = traverseNode(childNode, true, true);
            } else {
                childResult = traverseNode(childNode);
            }

            if (Array.isArray(childResult)) {
                result.push(...childResult);
            } else if (childResult && Object.keys(childResult).length > 0) {
                result.push(childResult);
            }
        }

        return result;
    } catch (error) {
        console.error("An error occurred in nodeChild:", error);
        return [];
    }
}

export function addAttributes(existingAttributes: any[] = [], newAttribute: any) {
    if (!newAttribute || !newAttribute.attribute) return existingAttributes;
    const existingAttributeIndex = existingAttributes.findIndex(attr => attr.attribute === newAttribute.attribute);

    if (existingAttributeIndex !== -1 && newAttribute.attribute === 'class') {
        // existingAttributes[existingAttributeIndex].value += ` ${newAttribute.value}`;
        existingAttributes[existingAttributeIndex].value = `'${existingAttributes[existingAttributeIndex].value.replace(/^'|'$/g, '').trim()} ${newAttribute.value.replace(/^'|'$/g, '').trim()}'`.trim();
    } else {
        existingAttributes.push(newAttribute);
    }
    return existingAttributes;
}

function processFlexData(flexData: any, currentAttributes: any) {

    if (flexData) {
        const flexJson = JSON.parse(flexData);
        let flexValue = 'oj-flex-item';

        if (flexJson.value) {
            flexValue += ` oj-sm-${flexJson.value}`;
        }
        if (flexJson.alignItems) {
            flexValue += ` oj-sm-align-items-${flexJson.alignItems}`;
        }
        if (flexJson.justifyContent) {
            flexValue += ` oj-sm-justify-content-${flexJson.justifyContent}`;
        }

        const flexObject = {
            "colon": true,
            "attribute": "class",
            "value": `'${flexValue}'`
        };

        return addAttributes(currentAttributes, flexObject);
    }

    return currentAttributes;
}

export async function fetchAllComponents(): Promise<void> {
    try {
        const [business, framework, spectra, core] = await Promise.all([
            businessComponentsFunc(),
            frameWorkSpectraComponentsFunc('frameWork'),
            frameWorkSpectraComponentsFunc('spectra'),
            coreComponentsFunc()
        ]);

        businessComponentsList = business;
        frameWorkComponentsList = framework;
        spectraComponentsList = spectra;
        coreComponentsList = core;
    } catch (error) {
        console.error('Error fetching components:', error);
    }
}