import { camelCase, cleanComponentProperties, lowerHyphen, upperUnderscore } from "./cleanFunc";
import { extractTextForButton } from "./components/button";
import { getComponentJson } from "./toolkitComponents";
import { businessComponentsList, traverseNode } from "./traverseNode";

// Helper function to transform a description into a type
function getTypeFromDescription(description: string): string {
    const words = description.trim().split(/\s+/);
    if (words.length === 1) {
        return words[0].toUpperCase();
    }
    return words.map(word => word[0]).join('').toUpperCase();
}

export function processOverviewComponent(node: any, projectForm: any, noListing: boolean) {
    // Transform type from the project form description
    const componentType = getTypeFromDescription(projectForm.componentDescription);

    // Process listing page properties if noListing is false
    let listingComponent = {};
    let tabHeader = "";
    let icon = "";
    let emptyListingComponent = ""

    if (!noListing) {
        // Extract the listing page and related components
        const listingPage = node.findOne((n: any) => n.name === 'Listing Page');
        const appNavigation = listingPage?.findOne((n: any) => n.name === 'In App Navigation');
        const tabsComponent = appNavigation?.children?.find((item: any) => item.name === 'Tabs');

        // Extract selected tab data
        const selectedTabData = tabsComponent?.children
            ?.map((tabNode: any) => cleanComponentProperties(tabNode.componentProperties))
            ?.find((tab: any) => tab.state === 'Selected') || {};
        tabHeader = lowerHyphen(selectedTabData.label || '');
        icon = selectedTabData.icon
            ? `oj-ux-ico-${figma.getNodeById(selectedTabData.icon)?.parent?.name || ''}`
            : '';

        // Empty listing page properties
        const emptyListingPage = node.findOne((n: any) => n.name === 'Listing Page (Empty)');
        const emptyListingComponentId = cleanComponentProperties(emptyListingPage?.componentProperties)?.listingbody;
        const emptyListingNode = figma.getNodeById(emptyListingComponentId) as ComponentNode;
        emptyListingComponent = lowerHyphen(emptyListingNode?.parent?.name || '');

        const listingPageProperties = cleanComponentProperties({
            ...listingPage?.componentProperties,
            ...Object.assign({}, ...listingPage?.exposedInstances?.map((instance: any) => instance.componentProperties))
        });
        const listingHeader = upperUnderscore(listingPageProperties.title || '');
        const hasListingMenu = listingPageProperties.listingmenu === 'True';

        const listingHeaderComponent = listingPage?.findOne((n: any) => n.name === 'Listing Header');
        const actionSlot = listingHeaderComponent?.findOne((n: any) => n.name === 'Action Slot');

        const secondaryButtonProps = cleanComponentProperties(
            actionSlot?.exposedInstances?.find((instance: any) => instance.name === 'Secondary Button')?.componentProperties
        );
        const secondaryActionLabel = upperUnderscore(secondaryButtonProps?.label || '');

        const listingMenuOptions = hasListingMenu
            ? (listingPage?.findOne((n: any) => n.name === 'More Actions Menu')?.children || [])
                .filter((menuNode: any) => menuNode.visible)
                .map((menuNode: any) => {
                    const menuProperties = cleanComponentProperties(menuNode.componentProperties);
                    return {
                        name: lowerHyphen(menuProperties.text || ''),
                        actionType: 'NAVIGATE',
                        showDivider: menuProperties.showdivider === 'true'
                    };
                })
            : [];

        const primaryButtonComponent = traverseNode(
            actionSlot?.exposedInstances?.find((instance: any) => instance.name === 'Primary Button')
        );
        const primaryActions = primaryButtonComponent
            ? [{
                actionType: 'NAVIGATE',
                name: lowerHyphen(primaryButtonComponent.nameholder || ''),
                buttonType: primaryButtonComponent.buttonName?.toUpperCase() || ''
            }]
            : [];

        listingComponent = {
            name: '',
            module: '',
            header: listingHeader,
            action: {
                secondaryActionsArray: [
                    {
                        label: secondaryActionLabel,
                        menuButton: hasListingMenu,
                        options: listingMenuOptions
                    }
                ],
                primaryActionsArray: primaryActions
            }
        };
    }

    // Header info component
    const headerInfoComponent = {
        header_info_component: {
            name: '',
            module: 'template-header-info'
        }
    };

    // Overview page properties
    const overviewPage = noListing ? node : node.findOne((n: any) => n.name === 'Overview Page');
    const overviewPageProperties = cleanComponentProperties({
        ...overviewPage?.componentProperties,
        ...Object.assign({}, ...overviewPage?.exposedInstances?.map((instance: any) => instance.componentProperties))
    });
    const overviewHeader = lowerHyphen(overviewPageProperties.title || '');

    // Context component from side panel
    const sidePanel = node.findOne((n: any) => n.name === 'Side Panel');
    const sideComponentName = camelCase(sidePanel?.children[0]?.name || '');
    const contextComponentJson = getComponentJson(businessComponentsList, sideComponentName);
    const contextComponent = {
        name: contextComponentJson?.name || '',
        module: contextComponentJson?.moduleName || ''
    };

    const hasOverviewMenu = overviewPageProperties.overviewmenu === 'True';

    // Overview action components
    const overviewHeaderComponent = overviewPage?.findOne((n: any) => n.name === 'Overview Header');
    const overviewActionSlot = overviewHeaderComponent?.findOne((n: any) => n.name === 'Action Slot');

    const overviewSecondaryButtonProps = cleanComponentProperties(
        overviewActionSlot?.exposedInstances?.find((instance: any) => instance.name === 'Secondary Button')?.componentProperties
    );
    const overviewSecondaryActionLabel = upperUnderscore(overviewSecondaryButtonProps?.label || '');

    const overviewMenuOptions = hasOverviewMenu
        ? (overviewPage?.findOne((n: any) => n.name === 'More Actions Menu')?.children || [])
            .filter((menuNode: any) => menuNode.visible)
            .map((menuNode: any) => {
                const menuProperties = cleanComponentProperties(menuNode.componentProperties);
                const iconComponent = menuNode.children?.find((item: any) => item.name === 'Icon');
                return {
                    name: lowerHyphen(menuProperties.text || ''),
                    actionType: 'NAVIGATE',
                    applicableFor: ['ACTIVE'],
                    icon: iconComponent ? `oj-ux-ico-${iconComponent?.children[0]?.name || ''}` : '',
                    showDivider: menuProperties.showdivider === 'true'
                };
            })
        : [];

    //fix primary button of overview
    const primaryButton = overviewActionSlot?.exposedInstances?.find((instance: any) => instance.name === 'Primary Button');
    const traversePrimary = traverseNode(primaryButton);
    const buttonTypeMapping: Record<string, string> = {
        'cta': 'PRIMARY',
        'outline': 'SECONDARY',
        'borderless': 'TERTIARY'
    };
    const buttonStyle = cleanComponentProperties(primaryButton?.componentProperties)?.style || '';
    const buttonName = buttonTypeMapping[buttonStyle] || 'PRIMARY';

    const overviewPrimaryActions = traversePrimary
        ? [{
            actionType: 'NAVIGATE',
            name: lowerHyphen(traversePrimary.nameholder || ''),
            buttonType: buttonName,
            icon: traversePrimary.imgSource || ''
        }]
        : [];

    // Central container components
    const centralContainer = node.findOne((n: any) => n.name === 'Overview Body');
    const componentList = centralContainer?.children
        ?.filter((child: any) => child.visible !== false)
        ?.map((child: any) => {
            const overviewLabel = child.children?.find((item: any) => item.type === 'TEXT')?.characters || '';
            const label = upperUnderscore(overviewLabel);

            const sectionProperties = cleanComponentProperties(child.componentProperties);
            const hideSectionHeader = sectionProperties?.sectionheader;

            const overviewComponent = child.children?.find((item: any) => item.type === 'INSTANCE');
            const componentName = camelCase(overviewComponent?.name || '');

            const componentJson = getComponentJson(businessComponentsList, componentName);
            return {
                label,
                hideSectionHeader,
                name: componentJson?.name || '',
                module: componentJson?.moduleName || ''
            };
        })
        ?.filter((component: any) => component.name && component.module);

    return {
        type: componentType,
        tabHeader,
        icon,
        noDataComp: emptyListingComponent,
        listing_component: noListing ? {} : listingComponent,
        header_info_component: headerInfoComponent,
        overview_component: {
            header: overviewHeader,
            context_component: contextComponent,
            action: {
                secondaryActionsArray: [
                    {
                        label: overviewSecondaryActionLabel,
                        menuButton: hasOverviewMenu,
                        options: overviewMenuOptions
                    }
                ],
                primaryActionsArray: overviewPrimaryActions,
                evaluator: ''
            },
            componentList
        }
    };
}

