import { getPropertiesDetails } from "./generateFunc";
import { idToKeyMapping24D } from "./maps/componentMap";
import { formDataFunc, mainFunction } from "./processFunc";
import { processOverviewComponent } from "./processOverview";
import { clearModals } from "./toolkitComponents";

interface CompProp {
    [key: string]: any;
}

interface Output {
    info: {
        metadataVersion: string;
        version: string;
        creationDate: number;
        lastModified: number;
        signature?: string;
    };
    componentType: string;
    flowType?: string;
    flowImplicitName?: string;
    reviewLayout?: string;
    reviewType?: string;
    finalRest?: {
        rests: any[];
        child: any[];
    };
    formLayout: {
        maxColumns: number;
        labelEdge: string;
        direction: string;
    };
    swaggerUrls: never[];
    components: {
        restFlow: any[];
        data: {
            externalMappedParams: any[];
            initRests: any[];
            css: string;
            customNLS: any;
            componentJson: any;
            componentHeader: string;
            stageName?: string;
            contracts?: any;
        };
        layout: CompProp[];
        componentName: string;
    }[];
    businessManifest?: {
        name: string;
        version: string;
        jetVersion: string;
        displayName: string;
        description: string;
        properties: any;
        methods: any;
        events: any;
        slots: any;
        id?: string;
        type?: string;
        componentName?: string;
    };
    componentManifest?: {
        name: string;
        moduleName?: string;
        version: string;
        jetVersion: string;
        displayName: string;
        description: string;
        properties: any;
        methods: any;
        events: any;
        slots: any;
    };
    businessComponent?: boolean;
    widgetManifest?: {
        componentName: string;
        module: string;
        segment: string[];
        width: {
            large: string;
            medium: string;
            small: string;
        };
        height?: number;
        'min-width'?: number;
        isWidget?: boolean
        widthRatio?: string
    };
}

const defaultOutput: Output = {
    info: {
        metadataVersion: "V3",
        version: "7.0.31-beta",
        creationDate: 1580109058384,
        lastModified: 1692706819721
    },
    componentType: "SINGLE_PAGE",
    formLayout: {
        maxColumns: 1,
        labelEdge: "top",
        direction: "row"
    },
    swaggerUrls: [],
    components: [
        {
            restFlow: [],
            data: {
                externalMappedParams: [],
                initRests: [],
                css: "",
                customNLS: {},
                componentJson: {},
                componentHeader: ""
            },
            layout: [],
            componentName: "interest-certificate"
        }
    ]
};

const defaultDrawerOutput: Output = {
    info: {
        metadataVersion: "V3",
        version: "7.0.31-beta",
        creationDate: 1580109058384,
        lastModified: 1692706819721
    },
    componentType: "SINGLE_PAGE",
    formLayout: {
        maxColumns: 1,
        labelEdge: "top",
        direction: "row"
    },
    swaggerUrls: [],
    components: [
        {
            restFlow: [],
            data: {
                externalMappedParams: [],
                initRests: [],
                css: "",
                customNLS: {},
                componentJson: {},
                componentHeader: ""
            },
            layout: [],
            componentName: "interest-certificate"
        }
    ]
};

function propertyConversion(node: any, input: any): CompProp {
    return Object.keys(input).reduce((acc: CompProp, key: string) => {
        // console.log(input[key]);
        // const type = Array.isArray(input[key]) ? 'string' : 'boolean';
        acc[key] = { description: key, type: 'boolean' };
        return acc;
    }, {});
}

function populateComponentManifest(
    projectForm: any,
    propertiesObject: CompProp | null = null
): any {
    return {
        name: projectForm?.componentName,
        moduleName: projectForm?.moduleName,
        version: "1.0.0",
        jetVersion: "^8.1.0",
        displayName: "Composite Component",
        description: "Composite Component",
        properties: propertiesObject || {
            params: {
                description: "Params Object",
                type: "object",
                writeback: true,
            }
        },
        methods: {},
        events: {},
        slots: {}
    };
}

function updateMetaDataForSinglePage(projectForm: any, node: any, output: Output) {
    if (node?.type !== 'INSTANCE') {
        console.log('Selected Component should be an Instance Node');
        figma.ui.postMessage({ error: 'export' });
        return;
    }
    const mainComponent = node.mainComponent as ComponentNode;
    const componentSetNode = mainComponent?.parent as ComponentSetNode;
    const propertiesDetails = componentSetNode?.type === 'COMPONENT_SET'
        ? getPropertiesDetails(componentSetNode)
        : mainComponent
            ? getPropertiesDetails(mainComponent)
            : null;
    const propertiesObject = propertiesDetails ? propertyConversion(node, propertiesDetails) : {};
    if (projectForm?.readOnly) {
        propertiesObject['readonly'] = { description: "Read Only", type: "boolean" };
    }

    if (propertiesObject?.readOnly) {
        delete propertiesObject.readOnly;
    }

    output.businessComponent = true;
    output.componentManifest = populateComponentManifest(projectForm, propertiesObject);
}


function metaDataUpdate(projectForm: any, node: any, output: Output) {
    output.componentType = projectForm?.componentType;
    output.components[0].componentName = projectForm.componentName || node.name;

    switch (projectForm?.componentType) {
        case 'SINGLE_PAGE':
            if (projectForm?.usableComponent) {
                updateMetaDataForSinglePage(projectForm, node, output);
            } else {
                output.componentManifest = populateComponentManifest(projectForm);
            }
            break;

        case 'FLOW':
            if (node?.type !== 'INSTANCE') {
                console.log('Selected Component should be an Instance Node');
                figma.ui.postMessage({ error: 'export' });
                return;
            }
            output.components[0].componentName = `${projectForm.componentName || node.name}-stage1`;
            output.components[0].data.componentHeader = projectForm.componentDescription || node.name;
            output.components[0].data.stageName = projectForm.componentDescription || node.name;
            output.flowType = 'Create';
            output.flowImplicitName = projectForm?.componentName;
            output.reviewLayout = '';
            output.reviewType = 'Default';
            output.finalRest = { rests: [], child: [] };
            break;

        case 'WIDGET':
            if (node?.type !== 'INSTANCE' && node.name !== 'Widget') {
                console.log('Selected Component should be an Instance Node');
                figma.ui.postMessage({ error: 'export' });
                return;
            }
            const size = node.componentProperties.size.value;
            const [height, minWidth] = size.split("x").map(Number);

            const heightMap = [0, 210, 420, 630];
            const minWidthMap = [0, 330, 660, 990];

            output.widgetManifest = {
                componentName: projectForm?.componentName,
                module: projectForm?.moduleName,
                segment: ["common"],
                width: { large: "4", medium: "6", small: "12" },
                height: heightMap[height],
                "min-width": minWidthMap[minWidth],
                isWidget: true,
                widthRatio: size
            };
            output.componentManifest = populateComponentManifest(projectForm);

            break;
    }
}

function flowContract(result: any, output: Output) {
    const contracts = result
        .filter((r: any) => r.type === 'Contract')
        .map((r: any) => ({
            buttonName: r?.buttonName,
            nameholder: r?.nameholder,
            selectionType: r?.selectionType,
            functionName: r?.nameholder.toLowerCase(),
        }));

    (output.components[0].data.contracts) = contracts;
}

function clearNodeCustomNLS(node: any, output: Output) {
    const jsonStr = node.getPluginData("customNLS");
    const customNLSArray = JSON.parse(jsonStr);
    const customNLSObject = customNLSArray.reduce((acc: any, obj: any) => {
        const key = Object.keys(obj)[0];
        acc[key] = obj[key];
        return acc;
    }, {});

    output.components[0].data.customNLS = customNLSObject;
    node?.setPluginData("customNLS", JSON.stringify([]));
}

const addSignature = async (jsonData: any, secretKey: string) => {
    try {
        const response = await fetch('http://localhost:3000/files/generateHash', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                "secretKey": secretKey,
                "jsonData": jsonData,
            })
        });
        if (!response.ok) {
            throw new Error(`HTTP error! Status: ${response.status}`);
        } else {
            const result = await response.json();
            const signature: string = await result.hashOutput
            return signature;
        }
    } catch (error) {
        console.error('Error in generate Hash', error);
    }
}

export async function metaData(node: any, projectForm: any) {

    if (projectForm?.componentType === 'OVERVIEW') {
        if (node?.type !== 'INSTANCE') {
            console.log('Selected Component should be an Instance Node');
            figma.ui.postMessage({ error: 'export' });
            return;
        }

        const mainComponent = node.mainComponent;
        const componentSet = mainComponent?.parent;

        const componentSetId = (componentSet?.type === 'COMPONENT_SET') ? componentSet.key : mainComponent?.key || null;

        if (!componentSetId) {
            console.log('Component Set ID not found');
            return null;
        }

        const componentName = idToKeyMapping24D(componentSetId);
        if (componentName !== 'overview' && componentName !== 'listingOverview') {
            console.log('Component is not valid for processing');
            return null;
        };
        const isOverviewPage = componentName === 'overview';
        const overviewOutput = processOverviewComponent(node, projectForm, isOverviewPage);
        console.log('Overview Output', overviewOutput);
        return overviewOutput;
    }


    if (projectForm) {
        metaDataUpdate(projectForm, node, defaultOutput);
    }
    // figma.ui.postMessage({ clearIgnoredList: true });

    const layout: CompProp[] = mainFunction(node, projectForm);
    if (projectForm?.componentType === 'FLOW') {
        defaultOutput.reviewType = formDataFunc()?.transactionReviewType || 'default';
        flowContract(layout, defaultOutput);
    }

    defaultOutput.components[0].layout = layout;
    clearNodeCustomNLS(node, defaultOutput);

    clearModals();

    addSignature(defaultOutput, "FigJetv1")
        .then((hashOutput: string | undefined) => {
            if (hashOutput !== undefined) {
                defaultOutput['info']['signature'] = hashOutput;
            } else {
                console.error('HashOutput is undefined');
            }
        })
        .catch((error: any) => {
            console.error('Error in generate Hash', error);
        });

    console.log('do', defaultOutput);
    return defaultOutput;
}


export function metaDataDrawer(node: any, projectForm: any) {
    if (projectForm) {
        metaDataUpdate(projectForm, node, defaultDrawerOutput);
    }

    const layout = mainFunction(node, projectForm);
    defaultDrawerOutput.components[0].layout = layout;

    addSignature(defaultDrawerOutput, "FigJetv1")
        .then((hashOutput: string | undefined) => {
            if (hashOutput !== undefined) {
                defaultDrawerOutput['info']['signature'] = hashOutput;
            } else {
                console.error('HashOutput is undefined');
            }
        })
        .catch((error: any) => {
            console.error('Error in generate Hash', error);
        });
    return defaultDrawerOutput;
}