import { camelCase, cleanComponentProperties } from "../cleanFunc";
import { setIgnoreKey } from "../pluginData";
import { traverseNode } from "../traverseNode";
import { extractTextFromNode } from "./commonFunc";

type Result = {
    description: string;
    value: string;
    state?: string;
    selected?: string;
};

export function inputReactions(node: any): string {
    try {
        if (!node) {
            throw new Error("The 'node' parameter is missing or undefined.");
        }

        const onClickReaction = node.reactions?.find((reaction: any) => reaction.trigger.type === 'ON_CLICK');
        if (!onClickReaction) return '';

        const destNode: any = figma.getNodeById(onClickReaction.action.destinationId);
        if (!destNode || !('children' in destNode)) return '';

        const destChild = destNode.children?.[0];
        const destGrand = destChild?.children?.[0];
        if (!destGrand || !('componentProperties' in destGrand)) return '';

        const componentProps = destGrand.componentProperties;
        const placeholder = Object.keys(componentProps)
            .find(key => key.includes('Label')) || '';

        setIgnoreKey(destNode.id);
        return placeholder ? String(componentProps[placeholder].value) : '';
    } catch (error) {
        console.error(error);
        return '';
    }
}

export function generateSelectSingleOptionList(node: any): Record<string, any> | undefined {
    const selectMenu = node.findOne((n: any) => n.name === "Menu");
    if (!selectMenu) return undefined;

    const options = selectMenu.children
        .map((child: any) => processGenericFunction(child))
        .filter((option: Result) => option.description !== '' && option.value !== '');

    return {
        optionSource: "custom",
        options: options,
        labelKey: "label",
        valueKey: "code"
    };
}

function formatValue(description: string): string {
    return `'${description.replace(/ /g, "").toUpperCase()}'`;
}

export function processGenericFunction(child: any, propertyExtractor?: (output: any) => any): Result {
    try {
        if (!child.visible) {
            return emptyResult();
        }

        let description: string = '';
        let sol: any;

        if (propertyExtractor) {
            const { output } = cleanComponentProperties(child.componentProperties);
            sol = output;
            description = propertyExtractor(sol) || '';
        } else {
            sol = extractTextFromNode(child, 'TEXT');
            description = sol || '';
        }

        if (!description) {
            throw new Error("Description not found or invalid");
        }

        const result: Result = {
            description,
            value: formatValue(description)
        };

        if (propertyExtractor) {
            result.state = sol.state?.value || '';
            result.selected = sol.selected?.value || '';
        }

        return result;
    } catch (error) {
        return handleError(error);
    }
}

function handleError(error: any): Result {
    console.error(error);
    return emptyResult();
}

function emptyResult(): Result {
    return {
        description: '',
        value: '',
        state: '',
        selected: ''
    };
}

export function generateSelectSingleExtraProperties(node: any): Record<string, any> {
    const reactionResult = selectDropDown(node);
    return reactionResult || {};
}

export function selectDropDown(node: any) {
    let destId = '';
    for (let i of node.reactions) {
        if (i.trigger.type === 'ON_CLICK') {
            destId = i.action.destinationId;
        }
    }
    const destNode: any = figma.getNodeById(destId);
    let output: any;
    if (destNode) {
        output = traverseNode(destNode);
        setIgnoreKey(destNode.id);
    }
    return output;
}

export function searchPlaceholder(node: any){
    const searchName = extractTextFromNode(node, 'TEXT');
    const nlsSearch = `nls.${searchName?.replace(/\s+/g, "")}`;
    return [{
        "attribute": "placeholder",
        "value": nlsSearch,
        "colon": true
    }]

}