import { camelCase, cleanComponentProperties } from "../cleanFunc";
import { setIgnoreKey } from "../pluginData";
import { traverseNode } from "../traverseNode";

type Result = {
    description: string;
    value: string;
    state?: string;
    selected?: string;
};

export function extractTextFromNode(node: any, criteria: string) {
    try {
        figma.skipInvisibleInstanceChildren = true;
        if (!node) {
            throw new Error("The 'node' parameter is missing or undefined. in extract Text from Node");
        }

        if (node.type === 'TEXT') {
            return node ? node.characters : '';
        }
        const textNode = node.findAllWithCriteria({ types: [criteria] })[0];
        return textNode ? textNode.characters : '';
    } catch (error) {
        console.error(error);
        return '';
    } finally {
        figma.skipInvisibleInstanceChildren = false;
    }
}

export function generateMappedValue(node: any) {
    const componentName = camelCase(
        node.name.replace(/\(.*?\)/g, '')  
            .replace(/[*\/❗️⏎&]/g, '')   
            .trim()
    );
    const lastFiveDigits = node.id.replace(/:/g, '').slice(-5);
    let output = `${componentName.replace(/[\s-]/g, '')}${lastFiveDigits}Var`;
    if (componentName === 'tabBar') {
        output = `selectedTab${lastFiveDigits}`
    } else if(node.type === 'TEXT') {
        output = `text${lastFiveDigits}`
    }
    return output;
}

export function connectedNodes(child: any, mappedName: string, getValue?: (output: any) => string) {
    try {
        const output = cleanComponentProperties(child.componentProperties);
        let description: string;
        if (getValue) {
            description = getValue(output);
        } else {
            description = extractTextFromNode(child, 'TEXT');
        }
        const value = `'${description?.replace(/ /g, "").toUpperCase()}'`;

        const destId = child.reactions.find((i: any) => i.trigger.type === 'ON_CLICK' && i.action.navigation === 'NAVIGATE')?.action.destinationId;
        const destNode: any = figma.getNodeById(destId);

        if (destNode) {
            const result: any = traverseNode(destNode);

            const conditionalExpression = { conditionalValue: `${mappedName}()===${value}` };
            if (Array.isArray(result) && result.length > 0) {
                result[0].conditionalExpression = conditionalExpression;
            } else if (result) {
                result.conditionalExpression = conditionalExpression;
            }

            setIgnoreKey(destNode.id);
            return result;
        }
    } catch (error) {
        console.error(error);
        return null;
    }
}

export function fetchOptions(child: any, getValue?: (output: any) => string) {
    if (!child) {
        console.error("The 'child' parameter is missing or undefined.");
        return null;
    }

    try {
        const output = cleanComponentProperties(child.componentProperties);
        let description: string;
        if (getValue) {
            description = getValue(output);
        } else {
            description = extractTextFromNode(child, 'TEXT');
        }
        const value = `'${description?.replace(/ /g, "").toUpperCase()}'`;
        const result: Result = {
            description,
            value
        }

        if (getValue) {
            result.state = output.state;
            result.selected = 'selected' in output ? output.selected : output.state;
        }

        return result;
    } catch (error) {
        console.error(error);
        return null;
    }
}